package test

import (
	"fmt"
	http_helper "github.com/gruntwork-io/terratest/modules/http-helper"
	"github.com/gruntwork-io/terratest/modules/random"
	"github.com/gruntwork-io/terratest/modules/terraform"
	"github.com/gruntwork-io/terratest/modules/test-structure"
	"strings"
	"testing"
	"time"
)

// Te wartości zastąp odpowiednimi dla Twoich modułów.
const dbDirProd = "../live/prod/data-stores/mysql"
const appDirProd = "../live/prod/services/hello-world-app"
const dbDirStage = "../live/stage/data-stores/mysql"
const appDirStage = "../live/stage/services/hello-world-app"

func TestHelloWorldAppStageWithStages(t *testing.T) {
	t.Parallel()

	// Wdrożenie bazy danych MySQL.
	defer test_structure.RunTestStage(t, "teardown_db", func() { teardownDb(t, dbDirStage) })
	test_structure.RunTestStage(t, "deploy_db", func() { deployDb(t, dbDirStage) })

	// Wdrożenie modułu hello-world-app.
	defer test_structure.RunTestStage(t, "teardown_app", func() { teardownHelloApp(t, appDirStage) })
	test_structure.RunTestStage(t, "deploy_app", func() { deployHelloApp(t, dbDirStage, appDirStage) })

	// Sprawdzenie poprawności działania modułu hello-world-app.
	test_structure.RunTestStage(t, "validate_app", func() { validateHelloApp(t, appDirStage) })
}

func TestHelloWorldAppProdWithStages(t *testing.T) {
	t.Parallel()

	// Wdrożenie bazy danych MySQL.
	defer test_structure.RunTestStage(t, "teardown_db", func() { teardownDb(t, dbDirProd) })
	test_structure.RunTestStage(t, "deploy_db", func() { deployDb(t, dbDirProd) })

	// Wdrożenie modułu hello-world-app.
	defer test_structure.RunTestStage(t, "teardown_app", func() { teardownHelloApp(t, appDirProd) })
	test_structure.RunTestStage(t, "deploy_app", func() { deployHelloApp(t, dbDirProd, appDirProd) })

	// Sprawdzenie poprawności działania modułu hello-world-app.
	test_structure.RunTestStage(t, "validate_app", func() { validateHelloApp(t, appDirProd) })
}

func createDbOpts(terraformDir string, uniqueId string) *terraform.Options {
	return &terraform.Options{
		TerraformDir: terraformDir,

		// Te moduły używają Terragrunt jako opakowania dla Terraform.
		TerraformBinary: "terragrunt",

		Vars: map[string]interface{}{
			"db_name":     fmt.Sprintf("test%s", uniqueId),
			"db_username": "password",
			"db_password": "password",
		},

		// terragrunt.hcl szuka ustawień za pomocą zmiennych środowiskowych.
		EnvVars: map[string]string{
			"TEST_STATE_DYNAMODB_TABLE": uniqueId,
		},
	}
}

func createHelloOpts(terraformDir string, uniqueId string) *terraform.Options {
	return &terraform.Options{
		TerraformDir: terraformDir,

		// Te moduły używają Terragrunt jako opakowania dla Terraform.
		TerraformBinary: "terragrunt",

		Vars: map[string]interface{}{
			"environment": fmt.Sprintf("test-%s", uniqueId),
		},

		// terragrunt.hcl szuka ustawień za pomocą zmiennych środowiskowych.
		EnvVars: map[string]string{
			"TEST_STATE_DYNAMODB_TABLE": uniqueId,
		},

		// W przypadku znanych błędów testy będą powtarzane do 3 razy,
    // w 5-sekundowych odstępach czasu.
		MaxRetries:         3,
		TimeBetweenRetries: 5 * time.Second,
		RetryableTerraformErrors: map[string]string{
			"RequestError: send request failed": "Throttling issue?",
		},
	}
}

func validateApp(t *testing.T, helloOpts *terraform.Options) {
	albDnsName := terraform.OutputRequired(t, helloOpts, "alb_dns_name")
	url := fmt.Sprintf("http://%s", albDnsName)

	maxRetries := 10
	timeBetweenRetries := 10 * time.Second

	http_helper.HttpGetWithRetryWithCustomValidation(
		t,
		url,
		nil,
		maxRetries,
		timeBetweenRetries,
		func(status int, body string) bool {
			return status == 200 && strings.Contains(body, "Hello, World")
		},
	)
}

func teardownDb(t *testing.T, dbDir string) {
	dbOpts := test_structure.LoadTerraformOptions(t, dbDir)
	defer terraform.Destroy(t, dbOpts)
}

func deployDb(t *testing.T, dbDir string) {
	uniqueId := random.UniqueId()
	dbOpts := createDbOpts(dbDir, uniqueId)

	// Zapisanie danych na dysku, aby inne etapy wykonywane później
  // miały możliwość odczytywania tych danych.
	test_structure.SaveTerraformOptions(t, dbDir, dbOpts)
	test_structure.SaveString(t, dbDir, "uniqueId", uniqueId)

	terraform.InitAndApply(t, dbOpts)
}

func teardownHelloApp(t *testing.T, helloAppDir string) {
	helloOpts := test_structure.LoadTerraformOptions(t, helloAppDir)
	defer terraform.Destroy(t, helloOpts)
}

func deployHelloApp(t *testing.T, dbDir string, helloAppDir string) {
	uniqueId := test_structure.LoadString(t, dbDir, "uniqueId")
	helloOpts := createHelloOpts(helloAppDir, uniqueId)

	// Zapisanie danych na dysku, aby inne etapy wykonywane później
  // miały możliwość odczytywania tych danych.
	test_structure.SaveTerraformOptions(t, helloAppDir, helloOpts)

	terraform.InitAndApply(t, helloOpts)
}

func validateHelloApp(t *testing.T, helloAppDir string) {
	helloOpts := test_structure.LoadTerraformOptions(t, helloAppDir)
	validateApp(t, helloOpts)
}
